#!/usr/bin/env python3
"""
===============================================================================
 AnthroHeart Permanent Archive Uploader (Arweave)
===============================================================================
Status:  READY FOR FUNDING
Author:  Thomas B. Sweet, Anthro Entertainment LLC
Purpose: Permanent, immutable storage of the Master Archive.
===============================================================================
"""

import os
import sys
import subprocess
import platform

# ---------------------------------------------------------------------------
# 1. VENV BOOTSTRAP
# ---------------------------------------------------------------------------

VENV_DIR = ".venv"
# 'arweave-python-client' is the standard library name often used, 
# but 'arweave-python' is likely what you meant. Adjusted for safety.
REQUIRED_PACKAGES = ["arweave-python", "pynacl", "tqdm"]

def in_venv():
    return sys.prefix != sys.base_prefix

def bootstrap_venv():
    if platform.system() == "Windows":
        venv_python = os.path.join(VENV_DIR, "Scripts", "python.exe")
    else:
        venv_python = os.path.join(VENV_DIR, "bin", "python")

    if not os.path.exists(VENV_DIR):
        print("🔧 Creating virtual environment...")
        subprocess.check_call([sys.executable, "-m", "venv", VENV_DIR])

    if not in_venv():
        print("📦 Installing Arweave dependencies...")
        subprocess.check_call([venv_python, "-m", "pip", "install", "--upgrade", "pip", "-q"])
        subprocess.check_call([venv_python, "-m", "pip", "install", *REQUIRED_PACKAGES, "-q"])
        print("🔁 Re-launching inside venv...\n")
        subprocess.check_call([venv_python] + sys.argv)
        sys.exit(0)

if not in_venv():
    bootstrap_venv()

# ---------------------------------------------------------------------------
# 2. LOGIC
# ---------------------------------------------------------------------------

import json
import hashlib
import time
from datetime import datetime, timezone
from nacl.signing import SigningKey
from nacl.encoding import HexEncoder
from arweave.arweave_lib import Wallet, Transaction
from arweave.transaction_upload import upload_transaction

# CONFIGURATION
FILE_PATH = "The_AnthroHeart_Collection_Bundle.7z"
WALLET_JSON = "arweave_wallet.json"
IDENTITY_KEY = "anthroheart_chain/anthro_identity.key" # From Genesis step
OUT_DIR = "anthroheart_chain"

# METADATA CONSTANTS
AUTHOR = "Thomas B. Sweet"
ORGANIZATION = "Anthro Entertainment LLC"
LICENSE = "CC0-1.0"
WEBSITE = "https://anthroentertainment.com"
GENESIS_DATE = "2025-12-25"

def get_hashes(filepath):
    print(f"⚙️  Hashing {filepath} (SHA256 & SHA512)...")
    sha256 = hashlib.sha256()
    sha512 = hashlib.sha512()
    
    total_size = os.path.getsize(filepath)
    processed = 0
    
    with open(filepath, "rb") as f:
        while chunk := f.read(1024 * 1024 * 16): # 16MB chunks
            sha256.update(chunk)
            sha512.update(chunk)
            processed += len(chunk)
            print(f"   Progress: {int((processed/total_size)*100)}%", end="\r")
            
    print("\n   Hashing complete.")
    return sha256.hexdigest(), sha512.hexdigest()

def load_identity_key(key_path):
    if not os.path.exists(key_path):
        print(f"❌ ERROR: Identity key not found at {key_path}")
        print("   Please run the Genesis Notarization script first to establish identity.")
        sys.exit(1)
        
    print(f"🔑 Loading identity from {key_path}...")
    with open(key_path, "r") as f:
        private_hex = f.read().strip()
    return SigningKey(private_hex, encoder=HexEncoder)

def main():
    # CHECKS
    if not os.path.exists(FILE_PATH):
        print(f"❌ Error: Archive not found: {FILE_PATH}")
        sys.exit(1)
    
    if not os.path.exists(WALLET_JSON):
        print(f"❌ Error: Arweave wallet not found: {WALLET_JSON}")
        print("   1. Create a wallet at https://arweave.app")
        print("   2. Save the keyfile here as 'arweave_wallet.json'")
        print("   3. Fund it with AR tokens.")
        sys.exit(1)

    # 1. LOAD IDENTITY (ED25519)
    # This is the key used to sign the content hash, verifying you are the author
    signing_key = load_identity_key(IDENTITY_KEY)
    public_key_hex = signing_key.verify_key.encode(encoder=HexEncoder).decode()

    # 2. HASH FILE
    h256, h512 = get_hashes(FILE_PATH)

    # 3. SIGN HASH
    # We sign the SHA512 hash as the primary content validator
    signature = signing_key.sign(h512.encode()).signature.hex()

    # 4. PREPARE METADATA
    metadata = {
        "title": "The AnthroHeart Collection (Master Archive)",
        "file_name": os.path.basename(FILE_PATH),
        "description": "Permanent public domain release of The AnthroHeart Collection.",
        "author": AUTHOR,
        "organization": ORGANIZATION,
        "license": LICENSE,
        "website": WEBSITE,
        "genesis_reference": {
            "date": GENESIS_DATE,
            "system": "Bitcoin/OpenTimestamps",
            "note": "This upload corresponds to the Genesis Record timestamped on Dec 25, 2025."
        },
        "integrity": {
            "sha256": h256,
            "sha512": h512,
            "signature_algorithm": "Ed25519",
            "signature_hex": signature,
            "signer_public_key": public_key_hex
        },
        "timestamp_upload": datetime.now(timezone.utc).isoformat()
    }

    # 5. INITIALIZE WALLET & ESTIMATE
    print("\n💰 Connecting to Arweave Network...")
    try:
        wallet = Wallet(WALLET_JSON)
        balance = wallet.balance
        print(f"   Wallet Balance: {balance} AR")
    except Exception as e:
        print(f"❌ Error reading wallet: {e}")
        sys.exit(1)

    # Create transaction object WITHOUT sending yet
    with open(FILE_PATH, "rb") as f:
        file_data = f.read()

    tx = Transaction(wallet, data=file_data)
    tx.add_tag("App-Name", "AnthroHeart-Archiver")
    tx.add_tag("App-Version", "1.0")
    tx.add_tag("Content-Type", "application/x-7z-compressed")
    tx.add_tag("Author", AUTHOR)
    tx.add_tag("License", LICENSE)
    tx.add_tag("SHA256", h256)
    
    # Calculate Cost
    reward = int(tx.reward) # in Winstons
    cost_ar = reward / 1000000000000
    
    print("\n" + "="*50)
    print(" UPLOAD COST ESTIMATION")
    print("="*50)
    print(f" File Size:  {os.path.getsize(FILE_PATH) / (1024*1024):.2f} MB")
    print(f" Cost:       {cost_ar:.6f} AR")
    print(f" Balance:    {balance:.6f} AR")
    print("-" * 50)

    if balance < cost_ar:
        print("❌ INSUFFICIENT FUNDS. Please top up your wallet.")
        sys.exit(1)

    confirm = input("⚠️  Type 'PAY' to sign and permanently upload this file: ")
    if confirm.strip().upper() != "PAY":
        print("Upload cancelled.")
        sys.exit(0)

    # 6. EXECUTE UPLOAD
    print("\n🚀 Signing and Uploading Data... (Do not close)")
    tx.sign()
    tx.send()
    
    # Wait a moment for propagation locally before dumping tags
    print(f"✅ DATA UPLOAD SENT. TXID: {tx.id}")

    # 7. UPLOAD METADATA (Small JSON, negligible cost)
    print("📝 Uploading Metadata Record...")
    meta_json = json.dumps(metadata, indent=2)
    meta_tx = Transaction(wallet, data=meta_json)
    meta_tx.add_tag("App-Name", "AnthroHeart-Metadata")
    meta_tx.add_tag("Content-Type", "application/json")
    meta_tx.add_tag("Linked-File-TX", tx.id)
    meta_tx.sign()
    meta_tx.send()

    print(f"✅ METADATA SENT. TXID:    {meta_tx.id}")

    # 8. SAVE RECEIPT
    receipt = {
        "file_tx": tx.id,
        "metadata_tx": meta_tx.id,
        "arweave_url": f"https://arweave.net/{tx.id}",
        "viewblock_url": f"https://viewblock.io/arweave/tx/{tx.id}",
        "upload_date": datetime.now(timezone.utc).isoformat(),
        "cost_ar": cost_ar
    }
    
    receipt_path = os.path.join(OUT_DIR, f"arweave_receipt_{datetime.now().strftime('%Y%m%d')}.json")
    with open(receipt_path, "w") as f:
        json.dump(receipt, f, indent=2)

    print("\n" + "="*60)
    print(" PERMANENT ARCHIVE COMPLETE")
    print("="*60)
    print(f"🌍 Permanent URL: https://arweave.net/{tx.id}")
    print(f"🧾 Receipt Saved: {receipt_path}")
    print("="*60)
    print("Note: Data may take 10-60 minutes to appear on gateways.")

if __name__ == "__main__":
    main()